function [En,psin,rhon]=Schrodinger(xmin,xmax,Ep,m,num_sol)
% Solve Schrodinger equation for energy eigenvalues and eigenfunctions.
% [energy, phi_fun]=solve_sch(length,number,potential,mass,sol_num)
% solves time-independent Schrodinger equation in region bounded by 0<=x<=length.
% Potential is infinite outside this region.
%
%   xmin			Limite infrieure du domaine (nm)
%   xmax            Limite suprieure du domaine (nm)
%   Ep				Energie potentielle (eV)
%   m               Masse de l'objet physique (par rapport  l'lectron)
%   num_sol         Nombre de solutions (valeurs propres) souhaites
%
%   En				Valeur propre (nergie tat confin) (eV)
%   rho_n           Densit de probabilit associ  la valeur propre En
%
% Olivier Pujol : septembre 2013
%
%% Constantes et conversions

h=6.6260755E-34;    % Constante de Planck (J.s)
hbar=h/(2*pi);
hbar2=hbar^2;
me=0.91093897E-30;  % Masse de l'lectron (kg)
eV=1.60217733E-19;  % Electron-volt (J) 
nm=1E-9;            % Nanomtre (m)

m=m*me;             % Conversion de la masse en valeur relle

n=length(Ep);       % Nombre de points d'chantillonnage
pas=(xmax-xmin)/n;  % Pas d'chantillonnage (nm)

cte=(hbar2)/(2*m*eV*(nm*pas)^2);    % Constante qui apparat dans l'quation de Shrodinger

clear hbar hbar2 m me eV nm

%% Construction de la matrice hamiltonienne

dia=nan(1,n-1);                     % Initialisation lment diagonal
nondia=nan(1,n-1);                  % Initialisation lment non diagonal

for i=2:n
    if min(Ep)<0
        dia(i-1)=2*cte+Ep(i)-min(Ep);           % Elment diagonal
    else
        dia(i-1)=2*cte+Ep(i);           % Elment diagonal
    end
    nondia(i-1)=cte;                % Elment non diagonal
end

d1=-nondia(2:n-1);							
Hamilt=diag(d1,1);                  % Diagonale haute de l'hamiltonien							
Hamilt=Hamilt+diag(d1,-1);          % Diagonale basse de l'hamiltonien

d2=dia(1:n-1);
Hamilt=Hamilt+diag(d2,0);           % Diagonale de l'hamiltonien

clear i cte dia nondia d1 d2 

%% Rsolution de l'quation aux valeurs propres H\psi = E\psi

[psi,E]=eigs(Hamilt,num_sol,'SM');			%use matlab function eigs to find num_sol eigenfunctions and eigenvalues

clear Hamilt

% Valeurs propres
En=nan(1,num_sol);
for i=1:num_sol
    En(i)=E(i,i)+min(Ep);										%return energy eigenvalues in vector e
end

clear E

% Vecteurs propres
psin=[zeros(1,num_sol);psi;zeros(1,num_sol)]; %wave function is zero at x = 0 and x = length

clear psi

%% Calcul de la densit de probabilit normalise

rhon=abs(psin).^2;

% % % clear psi_n

% normalisation
aire=nan(1,size(rhon,2));
for in=1:size(rhon,2)
    aire(in)=sum((rhon(1:end-1,in)+rhon(2:end,in))*pas)/2;    
    rhon(:,in)=rhon(:,in)/aire(in);
    psin(:,in)=psin(:,in)/sqrt(aire(in));
end

clear aire in pas

%% Partie graphique (trac de rho)

nt=input('Nombre de trac  effectuer (1 <= nt <= 7) ');

x=linspace(xmin,xmax,n);

if (nt> num_sol),error('Le nombre demand est suprieur au nombre de valeurs propres');end
if (nt<=0),error('Le nombre demand doit tre suprieur ou gal  1');end
if (nt>=7),error('Le nombre demand doit tre infrieur ou gal  7');end

for nfig=1:nt
    figure(nfig)
    subplot(2,1,1)
    plot(x,Ep,'linewidth',2);
    if (min(Ep)<0),axis([xmin xmax min(Ep) 0]);end
% % % %     if (min(Ep)==0),axis([xmin xmax min(Ep) max(Ep)]);end
    if (min(Ep)>=0),axis([xmin xmax min(Ep) Ep(end)]);end
    set(gca,'xminortick','on','yminortick','on');
    grid on;
    box on
    xlabel('{\itx} (nm)');
    title('{Energie potentielle (eV)}');
    %
    subplot(2,1,2)
    plot(x,rhon(1:end-1,nt-nfig+1),'r','linewidth',2);
    set(gca,'xminortick','on','yminortick','on');
    grid on;
    box on
    axis tight
    xlabel('{\itx} (nm)');
    title(['{\it{\rho_{p,',num2str(nfig),'}}}']);
end

clear ifig

if (nt==1),return;end

% Fabrication lgende pour figure suivante

Leg=cell(1,nt);
for il=1:nt
    Leg{il}=['{\it{\rho_{p,',num2str(il),'}}}'];
end
clear il
%
figure
subplot(2,1,1)
plot(x,Ep,'linewidth',2);
if (min(Ep)<0),axis([xmin xmax min(Ep) 0]);end
% % % % if (min(Ep)==0),axis([xmin xmax min(Ep) max(Ep)]);end
if (min(Ep)>=0),axis([xmin xmax min(Ep) Ep(end)]);end
set(gca,'xminortick','on','yminortick','on');
grid on;
box on
xlabel('{\itx} (nm)');
title('{Energie potentielle (eV)}');
%
subplot(2,1,2)
C=['r','b','k','g','m','y','c'];
for it=1:nt     
    plot(x,rhon(1:end-1,nt-it+1),C(it),'linewidth',2);
    hold on
end
legend(Leg)
set(gca,'xminortick','on','yminortick','on');
grid on;
box on
axis tight
xlabel('{\itx} (nm)');
title('{\it{\rho_p}}');

return